﻿using gov.va.med.vbecs.Common;
using System;
using System.Collections;
using System.Collections.Generic;
using System.Data;

namespace gov.va.med.vbecs.BOL
{
    /// <summary>
    /// Contains validation logic for automated instrument tests
    /// </summary>
    public static class TestValidation
    {
        private static Hashtable _resultLookup;

        private static Hashtable ResultLookup
        {
            get
            {
                if (_resultLookup == null)
                {
                    _resultLookup = new Hashtable();

                    // Empty results
                    _resultLookup.Add(string.Empty, "X");
                    _resultLookup.Add("X", "X");

                    // Positive results
                    _resultLookup.Add("1", "P");
                    _resultLookup.Add("2", "P");
                    _resultLookup.Add("3", "P");
                    _resultLookup.Add("4", "P");
                    _resultLookup.Add("P", "P");
                    _resultLookup.Add("Pos", "P");
                    _resultLookup.Add("Positive", "P");

                    // Negative results
                    _resultLookup.Add("0", "N");
                    _resultLookup.Add("N", "N");
                    _resultLookup.Add("Neg", "N");
                    _resultLookup.Add("Negative", "N");

                    _resultLookup.Add("U", "U");
                    _resultLookup.Add("NotProvided", "U");
                    _resultLookup.Add("I", "I");
                    _resultLookup.Add("Inconclusive", "I");
                    _resultLookup.Add("Y", "Y");
                    _resultLookup.Add("PooledRH", "Y");
                    _resultLookup.Add("Z", "Z");
                    _resultLookup.Add("Mx", "Mx");
                }

                return _resultLookup;
            }
        }

        private static Hashtable _xmInterpLookup;

        private static Hashtable xmInterpLookup
        {
            get
            {
                if (_xmInterpLookup == null)
                {
                    _xmInterpLookup = new Hashtable();

                    // Empty results
                    _xmInterpLookup.Add(string.Empty, "X");
                    _xmInterpLookup.Add("X", "X");

                    // Compatible results
                    _xmInterpLookup.Add("C", "Compatible");
                    _xmInterpLookup.Add("Comp", "Compatible");
                    _xmInterpLookup.Add("Compatible", "Compatible");

                    // Incompatible results
                    _xmInterpLookup.Add("I", "Incompatible");
                    _xmInterpLookup.Add("Incomp", "Incompatible");
                    _xmInterpLookup.Add("Incompatible", "Incompatible");
                }

                return _xmInterpLookup;
            }
        }

        /// <summary>
        /// Validaton of Unit ABO/Rh confirmation
        /// Implements TT_2.01 and TT_2.02
        /// </summary>
        /// <param name="AntiA"></param>
        /// <param name="AntiB"></param>
        /// <param name="AntiAB"></param>
        /// <param name="AntiD"></param>
        /// <param name="DControl"></param>
        /// <param name="ABOInterp"></param>
        /// <param name="RhInterp"></param>
        /// <param name="UnitABO"></param>
        /// <param name="UnitRh"></param>
        /// <param name="BloodUnitGuid"></param>
        /// <param name="unitStatusCode"></param>
        /// <returns></returns>
        public static TestValidationModel ValidateUnitABORh(string AntiA, string AntiB, string AntiAB, string AntiD,
            string DControl, string ABOInterp, string RhInterp, string UnitABO, string UnitRh, Guid? BloodUnitGuid,
            Common.UnitStatusCode unitStatusCode)
        {
            TestValidationModel validationResult = new TestValidationModel();

            // Collesce each value so we don't have any nulls
            AntiA = AntiA ?? "";
            AntiB = AntiB ?? "";
            AntiAB = AntiAB ?? "";
            AntiD = AntiD ?? "";
            DControl = DControl ?? "";
            ABOInterp = ABOInterp ?? "";
            RhInterp = RhInterp ?? "";
            UnitABO = UnitABO ?? "";
            UnitRh = UnitRh ?? "";

            if (!BloodUnitGuid.HasValue)
            {
                return new TestValidationModel(Common.TestValidationStatus.Unknown, null,
                    "Unknown until a Product Code is selected");
            }

            if (!ResultLookup.ContainsKey(AntiA.Trim()) ||
                !ResultLookup.ContainsKey(AntiB.Trim()) ||
                !ResultLookup.ContainsKey(AntiAB.Trim()) ||
                !ResultLookup.ContainsKey(AntiD.Trim()) ||
                !ResultLookup.ContainsKey(DControl.Trim()) ||
                !ResultLookup.ContainsKey(RhInterp.Trim()) ||
                !ResultLookup.ContainsKey(UnitRh.Trim()))
            {
                validationResult.TestValidationStatus = Common.TestValidationStatus.Error;
                validationResult.ErrorMessage = "An error occurred trying to validate this test.";
                return validationResult;
            }

            string systemABO, systemRh; //ABO and Rh interpretations that result from validation of test results

            AntiA = ResultLookup[AntiA.Trim()].ToString();
            AntiB = ResultLookup[AntiB.Trim()].ToString();
            AntiAB = ResultLookup[AntiAB.Trim()].ToString();
            AntiD = ResultLookup[AntiD.Trim()].ToString();
            DControl = ResultLookup[DControl.Trim()].ToString();
            ABOInterp = ABOInterp.Trim();
            RhInterp = ResultLookup[RhInterp.Trim()].ToString();
            UnitABO = UnitABO.Trim();
            UnitRh = ResultLookup[UnitRh.Trim()].ToString();

            validationResult.TestValidationStatus = Common.TestValidationStatus.Valid;
            validationResult.ExceptionForDisplayList = new List<ExceptionForDisplayModel>();
            validationResult.ErrorMessage = "";

            // BR_115.16
            if (CheckUnitEligibility(unitStatusCode) == false)
            {
                validationResult.TestValidationStatus = Common.TestValidationStatus.Error;
                validationResult.ErrorMessage += "Unit has been invalidated or is in a final status." +
                                                 Environment.NewLine;
                validationResult.ErrorMessage = validationResult.ErrorMessage.TrimEnd(Environment.NewLine.ToCharArray());
                return validationResult;
            }

            //BR_2.07 for ABO - Defect 290021
            if (UnitABO == "Mx")
            {
                validationResult.TestValidationStatus = Common.TestValidationStatus.Error;
                validationResult.ErrorMessage += Common.StrRes.ValidMsg.UC115.PooledABO().ResString;
                return validationResult;
            }

            //BR_2.07 for Rh - Defect 290021
            if (UnitRh == "Mx")
                UnitRh = "P";

            //calcualting systemABO
            if ((AntiA == "N" && AntiB == "N" && AntiAB == "N") ||
                (AntiA == "N" && AntiB == "N" && AntiAB == "X") ||
                (AntiA == "X" && AntiB == "X" && AntiAB == "N"))
            {
                systemABO = "O";
            }
            else if ((AntiA == "P" && AntiB == "N" && AntiAB == "P") ||
                     (AntiA == "P" && AntiB == "N" && AntiAB == "X"))
            {
                systemABO = "A";
            }
            else if ((AntiA == "N" && AntiB == "P" && AntiAB == "P") ||
                     (AntiA == "N" && AntiB == "P" && AntiAB == "X"))
            {
                systemABO = "B";
            }
            else if ((AntiA == "P" && AntiB == "P" && AntiAB == "P") ||
                     (AntiA == "P" && AntiB == "P" && AntiAB == "X"))
            {
                systemABO = "AB";
            }
            else
            {
                systemABO = "";
                validationResult.TestValidationStatus = Common.TestValidationStatus.Error;
                validationResult.ErrorMessage += "Invalid ABO." + Environment.NewLine;
            }

            //calcualting systemRh
            if ((AntiD == "P" && DControl == "N") ||
                (AntiD == "P" && DControl == "X"))
            {
                systemRh = "P";
            }
            else if ((AntiD == "N" && DControl == "N") ||
                     (AntiD == "N" && DControl == "X"))
            {
                systemRh = "N";
            }
            else if ((AntiD == "N" && DControl == "P") ||
                     (AntiD == "P" && DControl == "P") ||
                     (AntiD == "X" && DControl == "P"))
            {
                systemRh = "I";
            }
            else if (((AntiD == "X" && DControl == "N") ||
                      (AntiD == "X" && DControl == "X")) && UnitRh == "N")
            {
                systemRh = "";
                validationResult.TestValidationStatus = Common.TestValidationStatus.Error;
                validationResult.ErrorMessage += "Must perform Rh typing on Rh neg unit." + Environment.NewLine;
            }
            else if (AntiD == "X" && DControl == "X" && UnitRh == "P")
            {
                systemRh = "X";
            }
            else
            {
                systemRh = "";
                validationResult.TestValidationStatus = Common.TestValidationStatus.Error;
                validationResult.ErrorMessage += "Invalid Rh Typing." + Environment.NewLine;
            }

            //Defect 315252
            if (systemABO != "")
            {
                if (systemABO != ABOInterp)
                {
                    validationResult.TestValidationStatus = Common.TestValidationStatus.Error;
                    validationResult.ErrorMessage += "Invalid ABO." + Environment.NewLine;
                }
            }

            if (systemRh != "" && systemRh != "I")
            {
                if (systemRh != RhInterp && (DControl != "X" || RhInterp != UnitRh || AntiD == "X"))
                {
                    validationResult.TestValidationStatus = Common.TestValidationStatus.Error;
                    validationResult.ErrorMessage += "Rh interpretation does not match entries." + Environment.NewLine;
                }
            }
            else if (systemRh == "I" && validationResult.TestValidationStatus == Common.TestValidationStatus.Error)
            {
                validationResult.ErrorMessage += "Control is Positive. Continue?" + Environment.NewLine;
            }

            if (validationResult.TestValidationStatus != Common.TestValidationStatus.Error)
            {
                //BR_2.21
                if (systemRh == "I")
                {
                    validationResult.TestValidationStatus = Common.TestValidationStatus.Warning;
                    validationResult.ErrorMessage += "Control is Positive. Continue?" + Environment.NewLine;
                    //Defect 292497 302943
                    validationResult.ExceptionForDisplayList.Add(
                        new ExceptionForDisplayModel(ExceptionType.ABORh_confirmation_inconclusive_unit_quarantined,
                            "Inconclusive ABO/Rh",
                            Common.StrRes.ValidMsg.UC002.BR21ExceptionTitle().ResString,
                            "UT"));
                    validationResult.Quarantine = true;
                }
                //BR_2.17
                if (ABOInterp != UnitABO)
                {
                    validationResult.TestValidationStatus = Common.TestValidationStatus.Warning;
                    validationResult.ErrorMessage += "Interpretation does not match unit login information." +
                                                     Environment.NewLine;
                    //Defect 302943
                    validationResult.ExceptionForDisplayList.Add(
                        new ExceptionForDisplayModel(ExceptionType.ABORh_confirmation_does_not_match_login_ABORh,
                            "Abo/Rh Mismatch",
                            "Confirmation Abo/Rh does not match login Abo/Rh. The blood unit will be quarantined.",
                            "UT"));
                    validationResult.Quarantine = true;
                }
                else if (systemRh != "X" && systemRh != "I")
                {
                    if (RhInterp != UnitRh ||
                        systemRh != UnitRh)
                    {
                        validationResult.TestValidationStatus = Common.TestValidationStatus.Warning;
                        validationResult.ErrorMessage += "Does not match login ABO/Rh." +
                                                         Environment.NewLine;
                        //Defect 302943
                        validationResult.ExceptionForDisplayList.Add(
                            new ExceptionForDisplayModel(ExceptionType.ABORh_confirmation_does_not_match_login_ABORh,
                                "Abo/Rh Mismatch",
                                "Confirmation Abo/Rh does not match login Abo/Rh. The blood unit will be quarantined.",
                                "UT"));
                        validationResult.Quarantine = true;
                    }
                }

            }

            validationResult.ErrorMessage = validationResult.ErrorMessage.TrimEnd(Environment.NewLine.ToCharArray());

            if (validationResult.TestValidationStatus == TestValidationStatus.Valid &&
                string.IsNullOrWhiteSpace(validationResult.ErrorMessage))
            {
                validationResult.ErrorMessage = "Valid";
            }
            else if (validationResult.TestValidationStatus == Common.TestValidationStatus.Error)
            {
                validationResult.ExceptionForDisplayList.Clear();
                validationResult.UserWarningPromptList.Clear();
            }

            return validationResult;
        }

        /// <summary>
        /// Validation of patient ABO/Rh Typing
        /// TT_52.01, TT_52.02, TT_52.04
        /// </summary>
        /// <param name="AntiA"></param>
        /// <param name="AntiB"></param>
        /// <param name="AntiAB"></param>
        /// <param name="A1Cells"></param>
        /// <param name="BCells"></param>
        /// <param name="AntiD"></param>
        /// <param name="DControl"></param>
        /// <param name="ABOInterp"></param>
        /// <param name="RhInterp"></param>
        /// <param name="OrderedTest"></param>
        /// <param name="specimenExpiration"></param>
        /// <returns></returns>
        public static TestValidationModel ValidatePatientABORh(string AntiA, string AntiB, string AntiAB, string A1Cells,
            string BCells, string AntiD, string DControl, string ABOInterp, string RhInterp, BOL.OrderedTest OrderedTest,
            DateTime? specimenExpiration)
        {
            TestValidationModel validationResult = new TestValidationModel();

            // Get Patient Historic Abo and Rh
            var HistoricABO = "";
            var HistoricRh = "";

            if (OrderedTest != null &&
                OrderedTest.Patient != null)
            {
                HistoricABO = OrderedTest.Patient.AboRh.Abo.ToString();
                HistoricRh = OrderedTest.Patient.AboRh.RHFactorCode;
            }

            AntiA = AntiA ?? "";
            AntiAB = AntiAB ?? "";
            A1Cells = A1Cells ?? "";
            BCells = BCells ?? "";
            AntiD = AntiD ?? "";
            DControl = DControl ?? "";
            ABOInterp = ABOInterp ?? "";
            RhInterp = RhInterp ?? "";

            if (!ResultLookup.ContainsKey(AntiA.Trim()) ||
                !ResultLookup.ContainsKey(AntiB.Trim()) ||
                !ResultLookup.ContainsKey(AntiAB.Trim()) ||
                !ResultLookup.ContainsKey(A1Cells.Trim()) ||
                !ResultLookup.ContainsKey(BCells.Trim()) ||
                !ResultLookup.ContainsKey(AntiD.Trim()) ||
                !ResultLookup.ContainsKey(DControl.Trim()) ||
                !ResultLookup.ContainsKey(RhInterp.Trim()))
            {
                validationResult.TestValidationStatus = Common.TestValidationStatus.Error;
                validationResult.ErrorMessage = "An error occurred trying to validate this test.";
                return validationResult;
            }

            string systemABO, systemRh; //ABO and Rh interpretations that result from validation of test results
            string previousAboInterp = string.Empty;
            string previousRhInterp = string.Empty;
            DataTable dtPreviousResults = new DataTable();

            AntiA = ResultLookup[AntiA.Trim()].ToString();
            AntiB = ResultLookup[AntiB.Trim()].ToString();
            AntiAB = ResultLookup[AntiAB.Trim()].ToString();
            A1Cells = ResultLookup[A1Cells.Trim()].ToString();
            BCells = ResultLookup[BCells.Trim()].ToString();
            AntiD = ResultLookup[AntiD.Trim()].ToString();
            DControl = ResultLookup[DControl.Trim()].ToString();
            ABOInterp = ABOInterp.Trim();
            RhInterp = ResultLookup[RhInterp.Trim()].ToString();

            //setting HistoricABO and HistoricRh to empty string if patient is NR
            HistoricABO = HistoricABO.Trim() == "NA" ? "" : HistoricABO.Trim();
            HistoricRh = ResultLookup[HistoricRh.Trim()].ToString();
            HistoricRh = HistoricRh == "U" ? "" : HistoricRh;

            validationResult.TestValidationStatus = Common.TestValidationStatus.Valid;
            validationResult.ExceptionForDisplayList = new List<ExceptionForDisplayModel>();
            validationResult.ErrorMessage = "";

            // BR_92.09
            if (specimenExpiration.HasValue &&
                specimenExpiration.Value < BOL.VBECSDateTime.GetDivisionCurrentDateTime())
            {
                validationResult.ErrorMessage += Common.StrRes.ConfMsg.UC056.ExpiredTaskProcessed().ResString +
                                                 Environment.NewLine;

                validationResult.ExceptionForDisplayList.Add(
                    new ExceptionForDisplayModel(ExceptionType.Expired_task_processed,
                        "Expired Order Selected",
                        Common.StrRes.ConfMsg.UC056.ExpiredTaskProcessed().ResString,
                        "PT"));
            }

            //BR_92.06
            DataTable dtpatientAboRhHistory = OrderedTest.Patient.GetPatientAboRHHistory();

            DataRow[] aboHistory =
                dtpatientAboRhHistory.Select(Common.VbecsTables.SpecimenTest.BloodTestTypeId + " = 18 OR " +
                                             Common.VbecsTables.SpecimenTest.BloodTestTypeId + " = 65 OR " +
                                             Common.VbecsTables.SpecimenTest.BloodTestTypeId + " = 118",
                    Common.VbecsTables.SpecimenTest.TestDate + " DESC");
            DataRow[] rhHistory =
                dtpatientAboRhHistory.Select(Common.VbecsTables.SpecimenTest.BloodTestTypeId + " = 21 OR " +
                                             Common.VbecsTables.SpecimenTest.BloodTestTypeId + " = 68 OR " +
                                             Common.VbecsTables.SpecimenTest.BloodTestTypeId + " = 121",
                    Common.VbecsTables.SpecimenTest.TestDate + " DESC");
            if (aboHistory.Length > 0 && rhHistory.Length > 0)
            {
                //Skip if the most recent result was justified.
                if ((bool) aboHistory[0][Common.VbecsTables.SpecimenTest.AboRhChangeIndicator] == false)
                {
                    if (aboHistory.Length == 1 && rhHistory.Length == 1)
                    {
                        string abo = aboHistory[0][Common.VbecsTables.BloodUnitTest.TestResultId].ToString().Trim();
                        string rh = rhHistory[0][Common.VbecsTables.BloodUnitTest.TestResultId].ToString().Trim();

                        if (OrderedTest.Patient.ConversionAboRH.Abo != Common.ABO.NA &&
                            OrderedTest.Patient.ConversionAboRH.RH != Common.RH.NotProvided)
                        {
                            if (OrderedTest.Patient.ConversionAboRH.Abo != Common.Utility.GetAboFromString(abo) ||
                                OrderedTest.Patient.ConversionAboRH.RH != Common.Utility.GetRhFromString(rh))
                            {
                                validationResult.TestValidationStatus = Common.TestValidationStatus.Error;
                                validationResult.ErrorMessage +=
                                    Common.StrRes.SysErrMsg.UC092.ABORhMismatch().ResString + Environment.NewLine;
                                validationResult.ErrorMessage =
                                    validationResult.ErrorMessage.TrimEnd(Environment.NewLine.ToCharArray());
                                return validationResult;
                            }
                        }
                    }
                    else if (aboHistory.Length > 1 && rhHistory.Length > 1)
                    {
                        string aboNew = aboHistory[0][Common.VbecsTables.BloodUnitTest.TestResultId].ToString().Trim();
                        string rhNew = rhHistory[0][Common.VbecsTables.BloodUnitTest.TestResultId].ToString().Trim();
                        string aboOld = aboHistory[1][Common.VbecsTables.BloodUnitTest.TestResultId].ToString().Trim();
                        string rhOld = rhHistory[1][Common.VbecsTables.BloodUnitTest.TestResultId].ToString().Trim();
                        if (aboNew != "" &&
                            rhNew != "" &&
                            aboOld != "" &&
                            rhOld != "")
                        {
                            if (aboNew != aboOld ||
                                rhNew != rhOld)
                            {
                                validationResult.TestValidationStatus = Common.TestValidationStatus.Error;
                                validationResult.ErrorMessage +=
                                    Common.StrRes.SysErrMsg.UC092.ABORhMismatch().ResString + Environment.NewLine;
                                validationResult.ErrorMessage =
                                    validationResult.ErrorMessage.TrimEnd(Environment.NewLine.ToCharArray());
                                return validationResult;
                            }
                        }
                    }
                }
            }

            systemABO = InterpretPatientABO(AntiA, AntiB, AntiAB, A1Cells, BCells);
            systemRh = InterpretPatientRh(AntiD, DControl);

            if (systemABO == "X" || systemRh == "X")
            {
                validationResult.TestValidationStatus = Common.TestValidationStatus.Error;
                validationResult.ErrorMessage += "Missing results" + Environment.NewLine;
            }
            else
            {
                if (systemABO == ABOInterp &&
                    systemRh == RhInterp &&
                    (systemABO == HistoricABO || HistoricABO == string.Empty) &&
                    (systemRh == HistoricRh || HistoricRh == string.Empty))
                {
                    validationResult.TestValidationStatus = Common.TestValidationStatus.Valid;
                }
                //BR_115.27, BR_115.23
                else if (systemABO == ABOInterp &&
                         systemRh == RhInterp &&
                         OrderedTest.OrderableTest.IndexOf("Repeat") != -1 &&
                         OrderedTest.CheckForRepeatAboRhDiscrepancy(ABOInterp, RhInterp, ref previousAboInterp,
                             ref previousRhInterp, ref dtPreviousResults) == true)
                {
                    validationResult.TestValidationStatus = Common.TestValidationStatus.Warning;
                    validationResult.ExceptionForDisplayList.Add(
                        new ExceptionForDisplayModel(ExceptionType.ABORh_discrepancy,
                            "Discrepant Repeat ABO/Rh typing",
                            "Original and repeat ABO/Rh interpretation do not match. You must resolve the discrepancy before this specimen may be used. Continue to save?",
                            "PT"));
                    validationResult.ErrorMessage += "Discrepant Repeat ABO/Rh typing." + Environment.NewLine;
                }
                //BR_115.18
                else if (systemABO == ABOInterp &&
                         systemRh == RhInterp &&
                         (systemABO != HistoricABO || systemRh != HistoricRh))
                {
                    validationResult.TestValidationStatus = Common.TestValidationStatus.Warning;
                    validationResult.ExceptionForDisplayList.Add(
                        new ExceptionForDisplayModel(ExceptionType.ABORh_discrepancy,
                            "Discrepant ABO/Rh typing",
                            "Current ABO/Rh does not match previous results. Enter a comment to continue. Click Cancel to clear the ABO/Rh testing results and interpretation from the screen.",
                            "PT"));
                    validationResult.ErrorMessage += "Discrepant ABO/Rh typing." + Environment.NewLine;
                }
                else
                {
                    validationResult.TestValidationStatus = Common.TestValidationStatus.Error;
                    validationResult.ErrorMessage += "Interpretation does not match entries. Please correct." +
                                                     Environment.NewLine;
                }

                //BR_56.29
                if (ABOInterp != "" && HistoricABO != "" && RhInterp != "X" && HistoricRh != "X" &&
                    validationResult.TestValidationStatus != Common.TestValidationStatus.Error)
                {

                    if (ABOInterp != HistoricABO ||
                        RhInterp != HistoricRh)
                    {
                        DataTable dtIssuedUnits =
                            BOL.OrderedTest.GetEmergencyIssuedUnitsForOrderedTest(OrderedTest.OrderedTestGuid);
                        if (dtIssuedUnits.Rows.Count > 0)
                        {
                            validationResult.TestValidationStatus = Common.TestValidationStatus.Warning;
                            validationResult.ExceptionForDisplayList.Add(
                                new ExceptionForDisplayModel(ExceptionType.Unit_ER_issued_testing_problem,
                                    "ABO/Rh Mismatch",
                                    StrRes.SysErrMsg.UC056.EmergencyUnitsIssued().ResString,
                                    "PT"));
                            validationResult.ErrorMessage += "Units were emergency issued for this order." +
                                                             Environment.NewLine;
                        }
                    }
                }
            }

            validationResult.ErrorMessage = validationResult.ErrorMessage.TrimEnd(Environment.NewLine.ToCharArray());

            if (validationResult.TestValidationStatus == TestValidationStatus.Valid &&
                string.IsNullOrWhiteSpace(validationResult.ErrorMessage))
            {
                validationResult.ErrorMessage = "Valid";
            }
            else if (validationResult.TestValidationStatus == Common.TestValidationStatus.Error)
            {
                validationResult.ExceptionForDisplayList.Clear();
                validationResult.UserWarningPromptList.Clear();
            }

            return validationResult;
        }

        /// <summary>
        /// Interprets only ABO portion according to TT_52.01(A)
        /// Parameters passed to this method must be in P/N/X format
        /// </summary>
        /// <param name="AntiA"></param>
        /// <param name="AntiB"></param>
        /// <param name="AntiAB"></param>
        /// <param name="A1Cells"></param>
        /// <param name="BCells"></param>
        /// <returns></returns>
        private static string InterpretPatientABO(string AntiA, string AntiB, string AntiAB, string A1Cells, string BCells)
        {
            if (AntiA == "X" || AntiB == "X" || A1Cells == "X" || BCells == "X")
                return "X";

            if (AntiA == "N" &&
                AntiB == "N" &&
                (AntiAB == "X" || AntiAB == "N") &&
                A1Cells == "P" &&
                BCells == "P")
            {
                return "O";
            }
            else if (AntiA == "P" &&
                AntiB == "N" &&
                (AntiAB == "X" || AntiAB == "P") &&
                A1Cells == "N" &&
                BCells == "P")
            {
                return "A";
            }
            else if (AntiA == "N" &&
                AntiB == "P" &&
                (AntiAB == "X" || AntiAB == "P") &&
                A1Cells == "P" &&
                BCells == "N")
            {
                return "B";
            }
            else if (AntiA == "P" &&
                AntiB == "P" &&
                (AntiAB == "X" || AntiAB == "P") &&
                A1Cells == "N" &&
                BCells == "N")
            {
                return "AB";
            }
            else
            {
                return "I";
            }
        }

        /// <summary>
        /// Interprets only Rh portion according to TT_52.02(A)
        /// Parameters passed to this method must be in P/N/X format
        /// </summary>
        /// <param name="AntiD"></param>
        /// <param name="DControl"></param>
        /// <returns></returns>
        private static string InterpretPatientRh(string AntiD, string DControl)
        {
            if (AntiD == "X")
                return "X";

            if (AntiD == "P" &&
                (DControl == "N" || DControl == "X"))
            {
                return "P";
            }
            else if (AntiD == "N" &&
                (DControl == "N" || DControl == "X"))
            {
                return "N";
            }
            else
            {
                return "I";
            }
        }

        /// <summary>
        /// Validation of patient ABS Typing
        /// TT_53.01, TT_53.02
        /// </summary>
        /// <param name="SC1"></param>
        /// <param name="SC2"></param>
        /// <param name="SC3"></param>
        /// <param name="SC4"></param>
        /// <param name="SC5"></param>
        /// <param name="ABSInterp"></param>
        /// <param name="OrderedTest"></param>
        /// <param name="specimenExpiration"></param>
        /// <returns></returns>
        public static TestValidationModel ValidatePatientABS(string SC1, string SC2, string SC3, string SC4, string SC5,
            string ABSInterp, BOL.OrderedTest OrderedTest, DateTime? specimenExpiration)
        {
            TestValidationModel validationResult = new TestValidationModel();

            SC1 = SC1 ?? "";
            SC2 = SC2 ?? "";
            SC3 = SC3 ?? "";
            SC4 = SC4 ?? "";
            SC5 = SC5 ?? "";
            ABSInterp = ABSInterp ?? "";

            if (!ResultLookup.ContainsKey(SC1.Trim()) ||
                !ResultLookup.ContainsKey(SC2.Trim()) ||
                !ResultLookup.ContainsKey(SC3.Trim()) ||
                !ResultLookup.ContainsKey(SC4.Trim()) ||
                !ResultLookup.ContainsKey(SC5.Trim()) ||
                !ResultLookup.ContainsKey(ABSInterp.Trim()))
            {
                validationResult.TestValidationStatus = Common.TestValidationStatus.Error;
                validationResult.ErrorMessage = "An error occurred trying to validate this test.";
                return validationResult;
            }

            string systemABSInterp = "";

            SC1 = ResultLookup[SC1.Trim()].ToString();
            SC2 = ResultLookup[SC2.Trim()].ToString();
            SC3 = ResultLookup[SC3.Trim()].ToString();
            SC4 = ResultLookup[SC4.Trim()].ToString();
            SC5 = ResultLookup[SC5.Trim()].ToString();
            ABSInterp = ResultLookup[ABSInterp.Trim()].ToString();

            validationResult.TestValidationStatus = Common.TestValidationStatus.Valid;
            validationResult.ExceptionForDisplayList = new List<ExceptionForDisplayModel>();
            validationResult.ErrorMessage = "";

            // BR_92.09
            if (specimenExpiration.HasValue &&
                specimenExpiration.Value < BOL.VBECSDateTime.GetDivisionCurrentDateTime())
            {
                validationResult.ErrorMessage += Common.StrRes.ConfMsg.UC056.ExpiredTaskProcessed().ResString +
                                                 Environment.NewLine;

                validationResult.ExceptionForDisplayList.Add(
                    new ExceptionForDisplayModel(ExceptionType.Expired_task_processed,
                        "Expired Order Selected",
                        Common.StrRes.ConfMsg.UC056.ExpiredTaskProcessed().ResString,
                        "PT"));
            }

            if (SC1 == "X" || SC2 == "X")
            {
                validationResult.TestValidationStatus = Common.TestValidationStatus.Error;
                validationResult.ErrorMessage += "Missing results" + Environment.NewLine;
            }
            else
            {
                if (SC1 == "P" || SC2 == "P" || SC3 == "P" || SC4 == "P" || SC5 == "P")
                {
                    systemABSInterp = "P";
                }
                else
                {
                    systemABSInterp = "N";
                }

                if (systemABSInterp == ABSInterp)
                {
                    validationResult.TestValidationStatus = Common.TestValidationStatus.Valid;
                }
                else
                {
                    validationResult.TestValidationStatus = Common.TestValidationStatus.Error;
                    validationResult.ErrorMessage += "Interpretation does not match entries. Please correct." +
                                                     Environment.NewLine;
                }

                //BR_56.29
                if (ABSInterp == "P" &&
                    validationResult.TestValidationStatus != Common.TestValidationStatus.Error)
                {
                    DataTable dtIssuedUnits =
                        BOL.OrderedTest.GetEmergencyIssuedUnitsForOrderedTest(OrderedTest.OrderedTestGuid);
                    if (dtIssuedUnits.Rows.Count > 0)
                    {
                        validationResult.TestValidationStatus = Common.TestValidationStatus.Warning;
                        validationResult.ExceptionForDisplayList.Add(
                            new ExceptionForDisplayModel(ExceptionType.Unit_ER_issued_testing_problem,
                                "Positive ABS Result",
                                StrRes.SysErrMsg.UC056.EmergencyUnitsIssued().ResString,
                                "PT"));
                        validationResult.ErrorMessage += "Units were emergency issued for this order." +
                                                         Environment.NewLine;
                    }
                }
            }

            validationResult.ErrorMessage = validationResult.ErrorMessage.TrimEnd(Environment.NewLine.ToCharArray());

            if (validationResult.TestValidationStatus == TestValidationStatus.Valid &&
                string.IsNullOrWhiteSpace(validationResult.ErrorMessage))
            {
                validationResult.ErrorMessage = "Valid";
            }
            else if (validationResult.TestValidationStatus == Common.TestValidationStatus.Error)
            {
                validationResult.ExceptionForDisplayList.Clear();
                validationResult.UserWarningPromptList.Clear();
            }

            return validationResult;
        }

        /// <summary>
        /// Validation of patient DAT Typing
        /// TT_54.01
        /// </summary>
        /// <param name="Result"></param>
        /// <param name="DATInterp"></param>
        /// <param name="OrderedTest"></param>
        /// <param name="specimenExpiration"></param>
        /// <returns></returns>
        public static TestValidationModel ValidatePatientDAT(string Result, string DATInterp,
            BOL.OrderedTest OrderedTest, DateTime? specimenExpiration)
        {
            TestValidationModel validationResult = new TestValidationModel();

            Result = Result ?? "";
            DATInterp = DATInterp ?? "";

            if (!ResultLookup.ContainsKey(Result.Trim()) ||
                !ResultLookup.ContainsKey(DATInterp.Trim()))
            {
                validationResult.TestValidationStatus = Common.TestValidationStatus.Error;
                validationResult.ErrorMessage = "An error occurred trying to validate this test.";
                return validationResult;
            }

            Result = ResultLookup[Result.Trim()].ToString();
            DATInterp = ResultLookup[DATInterp.Trim()].ToString();

            validationResult.TestValidationStatus = Common.TestValidationStatus.Valid;
            validationResult.ExceptionForDisplayList = new List<ExceptionForDisplayModel>();
            validationResult.ErrorMessage = "";

            // BR_92.09
            if (specimenExpiration.HasValue &&
                specimenExpiration.Value < BOL.VBECSDateTime.GetDivisionCurrentDateTime())
            {
                validationResult.ErrorMessage += Common.StrRes.ConfMsg.UC056.ExpiredTaskProcessed().ResString +
                                                 Environment.NewLine;

                validationResult.ExceptionForDisplayList.Add(
                    new ExceptionForDisplayModel(ExceptionType.Expired_task_processed,
                        "Expired Order Selected",
                        Common.StrRes.ConfMsg.UC056.ExpiredTaskProcessed().ResString,
                        "PT"));
            }

            if (Result == "X" || DATInterp == "X")
            {
                validationResult.TestValidationStatus = Common.TestValidationStatus.Error;
                validationResult.ErrorMessage += "Missing results" + Environment.NewLine;
            }
            else
            {
                if (Result == DATInterp)
                {
                    validationResult.TestValidationStatus = Common.TestValidationStatus.Valid;
                }
                else
                {
                    validationResult.TestValidationStatus = Common.TestValidationStatus.Error;
                    validationResult.ErrorMessage += Common.StrRes.SysErrMsg.UC054.InvalidEntry().ResString +
                                                     Environment.NewLine;
                }

                //BR_56.29
                if (DATInterp == "P" &&
                    validationResult.TestValidationStatus != Common.TestValidationStatus.Error)
                {
                    DataTable dtIssuedUnits =
                        BOL.OrderedTest.GetEmergencyIssuedUnitsForOrderedTest(OrderedTest.OrderedTestGuid);
                    if (dtIssuedUnits.Rows.Count > 0)
                    {
                        validationResult.TestValidationStatus = Common.TestValidationStatus.Warning;
                        validationResult.ExceptionForDisplayList.Add(
                            new ExceptionForDisplayModel(ExceptionType.Unit_ER_issued_testing_problem,
                                "Positive DAT Result",
                                StrRes.SysErrMsg.UC056.EmergencyUnitsIssued().ResString,
                                "PT"));
                        validationResult.ErrorMessage += "Units were emergency issued for this order." +
                                                         Environment.NewLine;
                    }
                }
            }

            validationResult.ErrorMessage = validationResult.ErrorMessage.TrimEnd(Environment.NewLine.ToCharArray());
            if (validationResult.TestValidationStatus == TestValidationStatus.Valid &&
                string.IsNullOrWhiteSpace(validationResult.ErrorMessage))
            {
                validationResult.ErrorMessage = "Valid";
            }
            else if (validationResult.TestValidationStatus == Common.TestValidationStatus.Error)
            {
                validationResult.ExceptionForDisplayList.Clear();
                validationResult.UserWarningPromptList.Clear();
            }

            return validationResult;
        }

        /// <summary>
        /// Validation of crossmatch results
        /// TT_40.01, TT_40.02
        /// </summary>
        /// <param name="ISResult"></param>
        /// <param name="AHGResult"></param>
        /// <param name="ISInterp"></param>
        /// <param name="AHGInterp"></param>
        /// <param name="XMInterp"></param>
        /// <param name="BloodUnitGuid"></param>
        /// <param name="specimenExpiration"></param>
        /// <returns></returns>
        public static TestValidationModel ValidateXM(string ISResult, string AHGResult, string ISInterp,
            string AHGInterp, string XMInterp, Guid? BloodUnitGuid, DateTime? specimenExpiration)
        {
            TestValidationModel validationResult = new TestValidationModel();
            string systemXMInterp = "";

            if (!BloodUnitGuid.HasValue)
            {
                return new TestValidationModel(Common.TestValidationStatus.Unknown, null,
                    "Unknown until a Product Code is selected");
            }

            ISResult = ISResult ?? "";
            AHGResult = AHGResult ?? "";
            ISInterp = ISInterp ?? "";
            AHGInterp = AHGInterp ?? "";
            XMInterp = XMInterp ?? "";

            if (!ResultLookup.ContainsKey(ISResult.Trim()) ||
                !ResultLookup.ContainsKey(AHGResult.Trim()) ||
                !xmInterpLookup.ContainsKey(ISInterp.Trim()) ||
                !xmInterpLookup.ContainsKey(AHGInterp.Trim()) ||
                !xmInterpLookup.ContainsKey(XMInterp.Trim()))
            {
                validationResult.TestValidationStatus = Common.TestValidationStatus.Error;
                validationResult.ErrorMessage = "An error occurred trying to validate this test.";
                return validationResult;
            }

            ISResult = ResultLookup[ISResult.Trim()].ToString();
            AHGResult = ResultLookup[AHGResult.Trim()].ToString();
            ISInterp = xmInterpLookup[ISInterp.Trim()].ToString();
            AHGInterp = xmInterpLookup[AHGInterp.Trim()].ToString();
            XMInterp = xmInterpLookup[XMInterp.Trim()].ToString();

            validationResult.TestValidationStatus = Common.TestValidationStatus.Valid;
            validationResult.ExceptionForDisplayList = new List<ExceptionForDisplayModel>();
            validationResult.ErrorMessage = "";

            // BR_92.09
            if (specimenExpiration.HasValue &&
                specimenExpiration.Value < BOL.VBECSDateTime.GetDivisionCurrentDateTime())
            {
                validationResult.ErrorMessage += Common.StrRes.ConfMsg.UC056.ExpiredTaskProcessed().ResString +
                                                 Environment.NewLine;

                validationResult.ExceptionForDisplayList.Add(
                    new ExceptionForDisplayModel(ExceptionType.Expired_task_processed,
                        "Expired Order Selected",
                        Common.StrRes.ConfMsg.UC056.ExpiredTaskProcessed().ResString,
                        "PT"));
            }

            if (ISResult == "X" && AHGResult == "X")
            {
                validationResult.TestValidationStatus = Common.TestValidationStatus.Error;
                validationResult.ErrorMessage += "Missing results" + Environment.NewLine;
            }
            else
            {
                if ((ISResult == "X" && AHGResult == "N") ||
                    (ISResult == "N" && AHGResult == "N"))
                {
                    systemXMInterp = "Compatible";
                }
                else if ((ISResult == "X" && AHGResult == "P") ||
                         (ISResult == "N" && AHGResult == "P") ||
                         (ISResult == "P" && AHGResult == "P") ||
                         (ISResult == "P" && AHGResult == "N"))
                {
                    systemXMInterp = "Incompatible";
                }

                if (systemXMInterp != XMInterp)
                {
                    validationResult.TestValidationStatus = Common.TestValidationStatus.Error;
                    validationResult.ErrorMessage += Common.StrRes.SysErrMsg.UC040.InvalidEntry().ResString +
                                                     Environment.NewLine;
                }

                //BR_56.29
                if (XMInterp == "Incompatible" &&
                    validationResult.TestValidationStatus != Common.TestValidationStatus.Error)
                {
                    Common.UnitStatusCode statusCode = new BOL.BloodUnitStatus(BloodUnitGuid.Value).UnitStatusCode;
                    if (statusCode == Common.UnitStatusCode.Issued ||
                        statusCode == Common.UnitStatusCode.Transfused)
                    {
                        validationResult.TestValidationStatus = Common.TestValidationStatus.Warning;
                        validationResult.ExceptionForDisplayList.Add(
                            new ExceptionForDisplayModel(ExceptionType.Unit_ER_issued_testing_problem,
                                "Incompatible Crossmatch Result",
                                StrRes.SysErrMsg.UC056.EmergencyUnitsIssued().ResString,
                                "PT"));
                        validationResult.ErrorMessage += "Units were emergency issued for this order." +
                                                         Environment.NewLine;
                    }
                }
            }

            validationResult.ErrorMessage = validationResult.ErrorMessage.TrimEnd(Environment.NewLine.ToCharArray());
            if (validationResult.TestValidationStatus == TestValidationStatus.Valid &&
                string.IsNullOrWhiteSpace(validationResult.ErrorMessage))
            {
                validationResult.ErrorMessage = "Valid";
            }
            else if (validationResult.TestValidationStatus == Common.TestValidationStatus.Error)
            {
                validationResult.ExceptionForDisplayList.Clear();
                validationResult.UserWarningPromptList.Clear();
            }

            return validationResult;
        }

        /// <summary>
        /// Implements BR_41.12 for automated instrument
        /// </summary>
        /// <param name="BloodUnitGuid"></param>
        /// <param name="OrderableTestId"></param>
        /// <param name="AGInterp"></param>
        /// <returns></returns>
        private static bool AntigenTypingMismatch(Guid BloodUnitGuid, string OrderableTestId, string AGInterp)
        {
            bool mismatch = false;
            //
            DataTable testTypes = BOL.BloodTestType.GetBloodTestTypes(OrderableTestId);
            DataRow[] interpTest = testTypes.Select(Common.VbecsTables.BloodTestType.BloodTestName + " LIKE '*Interp'");
            if (interpTest.Length > 0)
            {
                int testTypeID = Convert.ToInt32(interpTest[0][Common.VbecsTables.BloodTestType.BloodTestTypeId]);
                //
                DataTable unitTests = BOL.BloodUnitTest.GetBloodUnitTestResult(BloodUnitGuid, testTypeID, Common.LogonUser.LogonUserDivisionCode);
                foreach (DataRow unitTest in unitTests.Rows)
                {
                    string testResult = unitTest[Common.VbecsTables.BloodUnitTest.TestResultId].ToString().Trim();
                    if ((testResult != "I") && (testResult != "") && (testResult != AGInterp.Trim()))
                    {
                        return true;
                    }
                }
                //
                DataTable antigenTests = BOL.AntibodyType.GetAntigenTest();
                DataRow[] matchingAntigenTest = antigenTests.Select("BloodTestTypeId = " + Convert.ToInt32(interpTest[0][Common.VbecsTables.BloodTestType.BloodTestTypeId]));
                if (matchingAntigenTest.Length > 0)
                {
                    int antigenTypeID = Convert.ToInt32(matchingAntigenTest[0][Common.VbecsTables.AntigenType.AntigenTypeId]);
                    DataTable unitAntigens = BOL.BloodUnitAntigen.GetBloodUnitAntigens(BloodUnitGuid, Common.LogonUser.LogonUserDivisionCode);
                    DataRow[] specificAntigens = unitAntigens.Select("AntigenTypeId = " + antigenTypeID);
                    foreach (DataRow unitAntigen in specificAntigens)
                    {
                        bool posResult = (bool)unitAntigen[Common.VbecsTables.BloodUnitAntigen.Positive];
                        if ((posResult && (AGInterp.Trim() == "N")) ||
                            (!posResult && (AGInterp.Trim() == "P")))
                        {
                            return true;
                        }
                    }
                }
            }
            //
            return mismatch;
        }

        /// <summary>
        /// Unit antigent typing validation
        /// TT_41.01
        /// </summary>
        /// <param name="Result"></param>
        /// <param name="AGInterp"></param>
        /// <param name="BloodUnitGuid"></param>
        /// <param name="OrderableTestId"></param>
        /// <param name="unitStatusCode"></param>
        /// <returns></returns>
        public static TestValidationModel ValidateUnitAntigentyping(string Result, string AGInterp, Guid? BloodUnitGuid,
            string OrderableTestId, Common.UnitStatusCode unitStatusCode)
        {
            if (!BloodUnitGuid.HasValue)
            {
                return new TestValidationModel(Common.TestValidationStatus.Unknown, null,
                    "Unknown until a Product Code is selected");
            }

            TestValidationModel validationResult = new TestValidationModel();

            // Collesce each value so we don't have any nulls
            Result = Result ?? "";
            AGInterp = AGInterp ?? "";
            OrderableTestId = OrderableTestId ?? "";

            if (!ResultLookup.ContainsKey(Result.Trim()) ||
                !ResultLookup.ContainsKey(AGInterp.Trim()))
            {
                validationResult.TestValidationStatus = Common.TestValidationStatus.Error;
                validationResult.ErrorMessage = "An error occurred trying to validate this test.";
                return validationResult;
            }

            Result = ResultLookup[Result.Trim()].ToString();
            AGInterp = ResultLookup[AGInterp.Trim()].ToString();

            validationResult.TestValidationStatus = Common.TestValidationStatus.Valid;
            validationResult.ExceptionForDisplayList = new List<ExceptionForDisplayModel>();
            validationResult.ErrorMessage = "";

            // BR_115.16
            if (CheckUnitEligibility(unitStatusCode) == false)
            {
                validationResult.TestValidationStatus = Common.TestValidationStatus.Error;
                validationResult.ErrorMessage += "Unit has been invalidated or is in a final status." +
                                                 Environment.NewLine;
                validationResult.ErrorMessage = validationResult.ErrorMessage.TrimEnd(Environment.NewLine.ToCharArray());
                return validationResult;
            }

            if (Result == "X" || AGInterp == "X")
            {
                validationResult.TestValidationStatus = Common.TestValidationStatus.Error;
                validationResult.ErrorMessage += "Missing results" + Environment.NewLine;
            }
            else
            {
                if (Result != AGInterp)
                {
                    validationResult.TestValidationStatus = Common.TestValidationStatus.Error;
                    validationResult.ErrorMessage += "Invalid Entry. Reenter your results." + Environment.NewLine;
                }

                if (validationResult.TestValidationStatus != Common.TestValidationStatus.Error)
                {
                    //BR_41.12
                    if (AntigenTypingMismatch(BloodUnitGuid.Value, OrderableTestId, AGInterp))
                    {
                        validationResult.TestValidationStatus = Common.TestValidationStatus.Warning;
                        validationResult.ErrorMessage +=
                            Common.StrRes.InfoMsg.UC041.AntigenTypingQuarantine().ResString + Environment.NewLine;
                        validationResult.Quarantine = true;
                        validationResult.UserWarningPromptList.Add(Common.StrRes.InfoMsg.UC041.AntigenTypingQuarantine());
                    }

                    //BR_56.29
                    if ((AGInterp == "P") &&
                        (BOL.AntigenTyping.PatientAntigenTypingMismatch(BloodUnitGuid.Value,
                            Convert.ToInt32(OrderableTestId.Trim()), AGInterp)))
                    {
                        Common.UnitStatusCode statusCode = new BOL.BloodUnitStatus(BloodUnitGuid.Value).UnitStatusCode;
                        if (statusCode == Common.UnitStatusCode.Issued ||
                            statusCode == Common.UnitStatusCode.Transfused)
                        {
                            validationResult.TestValidationStatus = Common.TestValidationStatus.Warning;
                            validationResult.ExceptionForDisplayList.Add(
                                new ExceptionForDisplayModel(ExceptionType.Unit_ER_issued_testing_problem,
                                    "Positive Antigen Typing Result",
                                    StrRes.SysErrMsg.UC056.EmergencyUnitsIssued().ResString,
                                    "PT"));
                            validationResult.ErrorMessage += "Units were emergency issued for this order." +
                                                             Environment.NewLine;
                        }
                    }
                }
            }

            validationResult.ErrorMessage = validationResult.ErrorMessage.TrimEnd(Environment.NewLine.ToCharArray());
            if (validationResult.TestValidationStatus == TestValidationStatus.Valid &&
                string.IsNullOrWhiteSpace(validationResult.ErrorMessage))
            {
                validationResult.ErrorMessage = "Valid";
            }
            else if (validationResult.TestValidationStatus == Common.TestValidationStatus.Error)
            {
                validationResult.ExceptionForDisplayList.Clear();
                validationResult.UserWarningPromptList.Clear();
            }

            return validationResult;
        }

        /// <summary>
        /// Validation of unit Weak D
        /// TT_41.02
        /// </summary>
        /// <param name="Result"></param>
        /// <param name="CC"></param>
        /// <param name="WeakDInterp"></param>
        /// <param name="BloodUnitGuid"></param>
        /// <param name="OrderableTestId"></param>
        /// <param name="UnitRh"></param>
        /// <param name="unitStatusCode"></param>
        /// <returns></returns>
        public static TestValidationModel ValidateUnitWeakD(string Result, string CC, string WeakDInterp,
            Guid? BloodUnitGuid, string OrderableTestId, string UnitRh, Common.UnitStatusCode unitStatusCode)
        {
            if (!BloodUnitGuid.HasValue)
            {
                return new TestValidationModel(Common.TestValidationStatus.Unknown, null,
                    "Unknown until a Product Code is selected");
            }

            TestValidationModel validationResult = new TestValidationModel();

            // Collesce each value so we don't have any nulls
            Result = Result ?? "";
            CC = CC ?? "";
            WeakDInterp = WeakDInterp ?? "";
            OrderableTestId = OrderableTestId ?? "";
            UnitRh = UnitRh ?? "";

            if (!ResultLookup.ContainsKey(Result.Trim()) ||
                !ResultLookup.ContainsKey(CC.Trim()) ||
                !ResultLookup.ContainsKey(WeakDInterp.Trim()) ||
                !ResultLookup.ContainsKey(UnitRh.Trim()))
            {
                validationResult.TestValidationStatus = Common.TestValidationStatus.Error;
                validationResult.ErrorMessage = "An error occurred trying to validate this test.";
                return validationResult;
            }

            Result = ResultLookup[Result.Trim()].ToString();
            CC = ResultLookup[CC.Trim()].ToString();
            WeakDInterp = ResultLookup[WeakDInterp.Trim()].ToString();
            UnitRh = ResultLookup[UnitRh.Trim()].ToString();

            string systemWeakDInterp = "";

            validationResult.TestValidationStatus = Common.TestValidationStatus.Valid;
            validationResult.ExceptionForDisplayList = new List<ExceptionForDisplayModel>();
            validationResult.ErrorMessage = "";

            // BR_115.16
            if (CheckUnitEligibility(unitStatusCode) == false)
            {
                validationResult.TestValidationStatus = Common.TestValidationStatus.Error;
                validationResult.ErrorMessage += "Unit has been invalidated or is in a final status." +
                                                 Environment.NewLine;
                validationResult.ErrorMessage = validationResult.ErrorMessage.TrimEnd(Environment.NewLine.ToCharArray());
                return validationResult;
            }

            if (Result == "X" || WeakDInterp == "X")
            {
                validationResult.TestValidationStatus = Common.TestValidationStatus.Error;
                validationResult.ErrorMessage += "Missing results" + Environment.NewLine;
            }
            else
            {
                if (Result == "P" && (CC == "X" || CC == "N"))
                {
                    systemWeakDInterp = "P";
                }
                else if (Result == "N" && (CC == "X" || CC == "N"))
                {
                    systemWeakDInterp = "N";
                }

                //BR_115.10
                if (UnitRh == "P")
                {
                    validationResult.TestValidationStatus = Common.TestValidationStatus.Error;
                    validationResult.ErrorMessage += "Cannot perform weak D testing on Rh positive unit." +
                                                     Environment.NewLine;
                }
                else if (systemWeakDInterp != WeakDInterp)
                {
                    validationResult.TestValidationStatus = Common.TestValidationStatus.Error;
                    validationResult.ErrorMessage += "Interpretation does not match results. Reenter your results." +
                                                     Environment.NewLine;
                }

                if (validationResult.TestValidationStatus != Common.TestValidationStatus.Error)
                {
                    //BR_41.22
                    if (WeakDInterp == "P" && UnitRh == "N")
                    {
                        validationResult.TestValidationStatus = Common.TestValidationStatus.Warning;
                        validationResult.ErrorMessage += Common.StrRes.InfoMsg.UC041.WeakDQuarantine().ResString +
                                                         Environment.NewLine;
                        validationResult.Quarantine = true;
                        validationResult.UserWarningPromptList.Add(Common.StrRes.InfoMsg.UC041.WeakDQuarantine());
                    }
                        //BR_41.12 if the unit is quaratined from previous step there is not need to check BR_41.12
                    else if (AntigenTypingMismatch(BloodUnitGuid.Value, OrderableTestId, WeakDInterp))
                    {
                        validationResult.TestValidationStatus = Common.TestValidationStatus.Warning;
                        validationResult.ErrorMessage +=
                            Common.StrRes.InfoMsg.UC041.AntigenTypingQuarantine().ResString + Environment.NewLine;
                        validationResult.Quarantine = true;
                        validationResult.UserWarningPromptList.Add(
                            Common.StrRes.InfoMsg.UC041.AntigenTypingQuarantine());
                    }


                    //BR_56.29
                    if ((WeakDInterp == "P") &&
                        (BOL.AntigenTyping.PatientAntigenTypingMismatch(BloodUnitGuid.Value,
                            Convert.ToInt32(OrderableTestId.Trim()), WeakDInterp)))
                    {
                        Common.UnitStatusCode statusCode =
                            new BOL.BloodUnitStatus(BloodUnitGuid.Value).UnitStatusCode;
                        if (statusCode == Common.UnitStatusCode.Issued ||
                            statusCode == Common.UnitStatusCode.Transfused)
                        {
                            validationResult.TestValidationStatus = Common.TestValidationStatus.Warning;
                            validationResult.ExceptionForDisplayList.Add(
                                new ExceptionForDisplayModel(ExceptionType.Unit_ER_issued_testing_problem,
                                    "Positive Antigen Typing Result",
                                    StrRes.SysErrMsg.UC056.EmergencyUnitsIssued().ResString,
                                    "PT"));
                            validationResult.ErrorMessage += "Units were emergency issued for this order." +
                                                             Environment.NewLine;
                        }
                    }
                }
            }

            validationResult.ErrorMessage = validationResult.ErrorMessage.TrimEnd(Environment.NewLine.ToCharArray());
            if (validationResult.TestValidationStatus == TestValidationStatus.Valid &&
                string.IsNullOrWhiteSpace(validationResult.ErrorMessage))
            {
                validationResult.ErrorMessage = "Valid";
            }
            else if (validationResult.TestValidationStatus == Common.TestValidationStatus.Error)
            {
                validationResult.ExceptionForDisplayList.Clear();
                validationResult.UserWarningPromptList.Clear();
            }

            return validationResult;
        }

        /// <summary>
        /// TT_41.01 Validation of antigen typing results for patient
        /// </summary>
        /// <param name="Result"></param>
        /// <param name="AGInterp"></param>
        /// <param name="PatientOrder"></param>
        /// <param name="specimenExpiration"></param>
        /// <returns></returns>
        public static TestValidationModel ValidatePatientAntigentyping(string Result, string AGInterp,
            BOL.OrderedTest PatientOrder, DateTime? specimenExpiration)
        {
            TestValidationModel validationResult = new TestValidationModel();

            // Collesce each value so we don't have any nulls
            Result = Result ?? "";
            AGInterp = AGInterp ?? "";

            if (!ResultLookup.ContainsKey(Result.Trim()) ||
                !ResultLookup.ContainsKey(AGInterp.Trim()))
            {
                validationResult.TestValidationStatus = Common.TestValidationStatus.Error;
                validationResult.ErrorMessage = "An error occurred trying to validate this test.";
                return validationResult;
            }

            Result = ResultLookup[Result.Trim()].ToString();
            AGInterp = ResultLookup[AGInterp.Trim()].ToString();

            validationResult.TestValidationStatus = Common.TestValidationStatus.Valid;
            validationResult.ExceptionForDisplayList = new List<ExceptionForDisplayModel>();
            validationResult.ErrorMessage = "";

            // BR_92.09
            if (specimenExpiration.HasValue &&
                specimenExpiration.Value < BOL.VBECSDateTime.GetDivisionCurrentDateTime())
            {
                validationResult.ErrorMessage += Common.StrRes.ConfMsg.UC056.ExpiredTaskProcessed().ResString +
                                                 Environment.NewLine;

                validationResult.ExceptionForDisplayList.Add(
                    new ExceptionForDisplayModel(ExceptionType.Expired_task_processed,
                        "Expired Order Selected",
                        Common.StrRes.ConfMsg.UC056.ExpiredTaskProcessed().ResString,
                        "PT"));
            }

            if (Result == "X" || AGInterp == "X")
            {
                validationResult.TestValidationStatus = Common.TestValidationStatus.Error;
                validationResult.ErrorMessage += "Missing results" + Environment.NewLine;
            }
            else
            {
                if (Result != AGInterp)
                {
                    validationResult.TestValidationStatus = Common.TestValidationStatus.Error;
                    validationResult.ErrorMessage += "Invalid Entry. Reenter your results" + Environment.NewLine;
                }

                if (validationResult.TestValidationStatus != Common.TestValidationStatus.Error)
                {
                    //BR_98.01
                    DataTable dtHistoricalResults = PatientOrder.GetHistoricalTestResultsForOrderedTest();
                    if (dtHistoricalResults.Rows.Count > 0)
                    {
                        DataRow[] dtHistoricalInterps =
                            dtHistoricalResults.Select(Common.VbecsTables.BloodTestType.BloodTestName +
                                                       " LIKE '*Interp*'");
                        if (dtHistoricalInterps.Length > 0)
                        {
                            foreach (DataRow drHistoricalInterp in dtHistoricalInterps)
                            {
                                if (AGInterp !=
                                    drHistoricalInterp[Common.VbecsTables.TestResult.TestResultId].ToString().Trim() &&
                                    drHistoricalInterp[Common.VbecsTables.TestResult.TestResultId].ToString().Trim() !=
                                    string.Empty)
                                {
                                    validationResult.TestValidationStatus = Common.TestValidationStatus.Warning;
                                    validationResult.ErrorMessage +=
                                        Common.StrRes.Tooltips.UC098.HistoricalMismatch().ResString +
                                        Environment.NewLine;
                                    validationResult.ExceptionForDisplayList.Add(
                                        new ExceptionForDisplayModel(ExceptionType.Discrepant_patient_antigen_typing,
                                            "Discrepant Antigen Typing",
                                            StrRes.ConfMsg.UC098.HistoricalMismatch().ResString,
                                            "PT"));
                                    break;
                                }
                            }
                        }
                    }
                }
            }

            validationResult.ErrorMessage = validationResult.ErrorMessage.TrimEnd(Environment.NewLine.ToCharArray());
            if (validationResult.TestValidationStatus == TestValidationStatus.Valid &&
                string.IsNullOrWhiteSpace(validationResult.ErrorMessage))
            {
                validationResult.ErrorMessage = "Valid";
            }
            else if (validationResult.TestValidationStatus == Common.TestValidationStatus.Error)
            {
                validationResult.ExceptionForDisplayList.Clear();
                validationResult.UserWarningPromptList.Clear();
            }

            return validationResult;
        }

        /// <summary>
        /// Validation of patient Weak D
        /// TT_41.02
        /// </summary>
        /// <param name="Result"></param>
        /// <param name="CC"></param>
        /// <param name="WeakDInterp"></param>
        /// <param name="OrderedTest"></param>
        /// <param name="specimenExpiration"></param>
        /// <returns></returns>
        public static TestValidationModel ValidatePatientWeakD(string Result, string CC, string WeakDInterp,
            BOL.OrderedTest OrderedTest, DateTime? specimenExpiration)
        {
            TestValidationModel validationResult = new TestValidationModel();

            string PatientRh = OrderedTest.Patient.AboRh.RHFactorCode;

            // Collesce each value so we don't have any nulls
            Result = Result ?? "";
            CC = CC ?? "";
            WeakDInterp = WeakDInterp ?? "";

            if (!ResultLookup.ContainsKey(Result.Trim()) ||
                !ResultLookup.ContainsKey(CC.Trim()) ||
                !ResultLookup.ContainsKey(WeakDInterp.Trim()) ||
                !ResultLookup.ContainsKey(PatientRh.Trim()))
            {
                validationResult.TestValidationStatus = Common.TestValidationStatus.Error;
                validationResult.ErrorMessage = "An error occurred trying to validate this test.";
                return validationResult;
            }

            Result = ResultLookup[Result.Trim()].ToString();
            CC = ResultLookup[CC.Trim()].ToString();
            WeakDInterp = ResultLookup[WeakDInterp.Trim()].ToString();
            PatientRh = ResultLookup[PatientRh.Trim()].ToString();

            string systemWeakDInterp = "";

            validationResult.TestValidationStatus = Common.TestValidationStatus.Valid;
            validationResult.ExceptionForDisplayList = new List<ExceptionForDisplayModel>();
            validationResult.ErrorMessage = "";

            // BR_92.09
            if (specimenExpiration.HasValue &&
                specimenExpiration.Value < BOL.VBECSDateTime.GetDivisionCurrentDateTime())
            {
                validationResult.ErrorMessage += Common.StrRes.ConfMsg.UC056.ExpiredTaskProcessed().ResString +
                                                 Environment.NewLine;

                validationResult.ExceptionForDisplayList.Add(
                    new ExceptionForDisplayModel(ExceptionType.Expired_task_processed,
                        "Expired Order Selected",
                        Common.StrRes.ConfMsg.UC056.ExpiredTaskProcessed().ResString,
                        "PT"));
            }

            if (Result == "X" || WeakDInterp == "X")
            {
                validationResult.TestValidationStatus = Common.TestValidationStatus.Error;
                validationResult.ErrorMessage += "Missing results" + Environment.NewLine;
            }
            else
            {
                if (Result == "P" && (CC == "X" || CC == "N"))
                {
                    systemWeakDInterp = "P";
                }
                else if (Result == "N" && (CC == "X" || CC == "N"))
                {
                    systemWeakDInterp = "N";
                }

                //BR_98.02
                if (PatientRh == "P")
                {
                    validationResult.TestValidationStatus = Common.TestValidationStatus.Error;
                    validationResult.ErrorMessage += "Cannot perform weak D testing on Rh positive patient." +
                                                     Environment.NewLine;
                }
                else if (systemWeakDInterp != WeakDInterp)
                {
                    validationResult.TestValidationStatus = Common.TestValidationStatus.Error;
                    validationResult.ErrorMessage += "Interpretation does not match results. Reenter your results." +
                                                     Environment.NewLine;
                }

                if (validationResult.TestValidationStatus != Common.TestValidationStatus.Error)
                {
                    //BR_98.01
                    DataTable dtHistoricalResults = OrderedTest.GetHistoricalTestResultsForOrderedTest();
                    if (dtHistoricalResults.Rows.Count > 0)
                    {
                        DataRow[] dtHistoricalInterps =
                            dtHistoricalResults.Select(Common.VbecsTables.BloodTestType.BloodTestName +
                                                       " LIKE '*Interp*'");
                        if (dtHistoricalInterps.Length > 0)
                        {
                            foreach (DataRow drHistoricalInterp in dtHistoricalInterps)
                            {
                                if (WeakDInterp !=
                                    drHistoricalInterp[Common.VbecsTables.TestResult.TestResultId].ToString().Trim() &&
                                    drHistoricalInterp[Common.VbecsTables.TestResult.TestResultId].ToString().Trim() !=
                                    string.Empty)
                                {
                                    validationResult.TestValidationStatus = Common.TestValidationStatus.Warning;
                                    validationResult.ErrorMessage +=
                                        Common.StrRes.Tooltips.UC098.HistoricalMismatch().ResString +
                                        Environment.NewLine;
                                    validationResult.ExceptionForDisplayList.Add(
                                        new ExceptionForDisplayModel(ExceptionType.Discrepant_patient_antigen_typing,
                                            "Discrepant Antigen Typing",
                                            StrRes.ConfMsg.UC098.HistoricalMismatch().ResString,
                                            "PT"));
                                    break;
                                }
                            }
                        }
                    }
                }
            }

            validationResult.ErrorMessage = validationResult.ErrorMessage.TrimEnd(Environment.NewLine.ToCharArray());

            if (validationResult.TestValidationStatus == TestValidationStatus.Valid &&
                string.IsNullOrWhiteSpace(validationResult.ErrorMessage))
            {
                validationResult.ErrorMessage = "Valid";
            }
            else if (validationResult.TestValidationStatus == Common.TestValidationStatus.Error)
            {
                validationResult.ExceptionForDisplayList.Clear();
                validationResult.UserWarningPromptList.Clear();
            }

            return validationResult;
        }

        /// <summary>
        /// Checks if the unit is eligible for testing
        /// BR_5.03
        /// </summary>
        /// <param name="unitStatusCode"></param>
        /// <returns></returns>
        private static bool CheckUnitEligibility(Common.UnitStatusCode unitStatusCode)
        {
            if (unitStatusCode == Common.UnitStatusCode.Transferred ||
                unitStatusCode == Common.UnitStatusCode.Transfused ||
                unitStatusCode == Common.UnitStatusCode.Modified ||
                unitStatusCode == Common.UnitStatusCode.Discarded)
            {
                return false;
            }

            return true;
        }

    }
}
